#include "stdafx.h"
#include "Debug.h"
#include "Common.h"
#include "Application.h"

#include "TLR_Results.h"
#include "PNM_PNIOAPCFG_Public.h"
#include "PNM_PNIOAPCTL_Public.h"
#include "rcX_Public.h"
#include <stdio.h>
#include <time.h>

/*************************************************************************************************
 * @brief This method handles keys pressed by the user.
 * 
 * @param ptApp Pointer to application data.
 * @param iKey is the key pressed.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandleKey(APPLICATION_T*  ptApp, TLR_INT iKey)
{
  switch(iKey)
  {
    case 'h':
		App_ShowHelp();
		break;
	case 's':
		App_SendBusScan(ptApp);
		break;
	case 'r':
		App_SendReadReq(ptApp);
		break;
	case 'R':
		App_SendReadImplReq(ptApp);
		break;
	case 'w':
		App_SendWriteReq(ptApp);
		break;
	case 'g':
		App_GetHandles(ptApp);
		break;
	case 'i':
		App_ReadOutputData(ptApp);
		break;
	case 'f':
		App_SendFactoryResetReq(ptApp);
		break;
	default:
		DEBUG("Unknown key  (%c)  pressed.\n", iKey);
		break;
  }
}


/******************************************************************/
/*             functions sending requests to firmware             */
/******************************************************************/

void App_SendBusScan(APPLICATION_T* ptApp)
{
	APIOC_DCP_IDENT_ALL_REQ_T tPck;

	memset(&tPck, 0, sizeof(tPck));
	
	tPck.tHead.ulCmd  = PNIO_APCTL_CMD_DCP_IDENT_ALL_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	tPck.tData.ulTimeout = 6;
	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_SendReadReq(APPLICATION_T* ptApp)
{
	APIOC_READ_REQ_T tPck;

	char cBuf[128];

	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = PNIO_APCTL_CMD_READ_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	tPck.tData.ulDataLength = 1024;

	tPck.tData.ulHandle = 0x1000;
	while (tPck.tData.ulHandle > (ptApp->tCommon.ulConfigIodAmount -1))
	{
		DEBUG("From which IO-Device shall data be read? Enter handle: ");
		fgets(cBuf, 128, stdin);
		tPck.tData.ulHandle = atol(cBuf);
	}

	DEBUG("What API shall be used? Enter API: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.ulApi = atol(cBuf);

	DEBUG("What slot shall be used? Enter slot: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usSlot = atoi(cBuf);

	DEBUG("What subslot shall be used? Enter subslot: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usSubSlot = atoi(cBuf);

	DEBUG("What index shall be read? Enter index: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usIndex = atoi(cBuf);


	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_SendReadImplReq(APPLICATION_T* ptApp)
{
	APIOC_READ_IMPL_REQ_T tPck;

	char cBuf[128];

	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = PNIO_APCTL_CMD_READ_IMPL_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	tPck.tData.ulDataLength = 1024;

	DEBUG("Specify IP-address of device to read from: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.ulIPAddress = atol(cBuf);

	DEBUG("Specify VendorId of device to read from: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usVendorId = atoi(cBuf);

	DEBUG("Specify DeviceId of device to read from: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usDeviceId = atoi(cBuf);

	DEBUG("Specify InstanceId of device to read from: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usInstanceId = atoi(cBuf);

	DEBUG("What API shall be used? Enter API: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.ulApi = atol(cBuf);

	DEBUG("What slot shall be used? Enter slot: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usSlot = atoi(cBuf);

	DEBUG("What subslot shall be used? Enter subslot: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usSubSlot = atoi(cBuf);

	DEBUG("What index shall be read? Enter index: ");
	fgets(cBuf, 128, stdin);
	tPck.tData.usIndex = atoi(cBuf);


	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_SendWriteReq(APPLICATION_T* ptApp)
{
	CIFX_PACKET tCifX;
	APIOC_WRITE_REQ_T* ptPck = (APIOC_WRITE_REQ_T*)&tCifX;

	char cBuf[128];

	memset(ptPck, 0, sizeof(*ptPck));

	ptPck->tHead.ulCmd  = PNIO_APCTL_CMD_WRITE_REQ;
	ptPck->tHead.ulDest = 0x20;
	ptPck->tHead.ulLen  = sizeof(ptPck->tData);
	
	ptPck->tData.usAlign = 0;

	ptPck->tData.ulHandle = 0x1000;
	while (ptPck->tData.ulHandle > (ptApp->tCommon.ulConfigIodAmount -1))
	{
		DEBUG("To which IO-Device shall data be written? Enter handle: ");
		fgets(cBuf, 128, stdin);
		ptPck->tData.ulHandle = atol(cBuf);
	}

	DEBUG("What API shall be used? Enter API: ");
	fgets(cBuf, 128, stdin);
	ptPck->tData.ulApi = atol(cBuf);

	DEBUG("What slot shall be used? Enter slot: ");
	fgets(cBuf, 128, stdin);
	ptPck->tData.usSlot = atoi(cBuf);

	DEBUG("What subslot shall be used? Enter subslot: ");
	fgets(cBuf, 128, stdin);
	ptPck->tData.usSubSlot = atoi(cBuf);

	DEBUG("What index shall be written? Enter index: ");
	fgets(cBuf, 128, stdin);
	ptPck->tData.usIndex = atoi(cBuf);

	DEBUG("How much data shall be written (data needs to manually be entered afterwards!)? Enter amount: ");
	ptPck->tData.ulDataLength = 1111;
	while (ptPck->tData.ulDataLength > 10)
	{
		fgets(cBuf, 128, stdin);
		ptPck->tData.ulDataLength = atol(cBuf);
		if (ptPck->tData.ulDataLength > 10)
		{
			DEBUG("Enter amount < 10. How much data shall be written? Enter amount: ");
		}
	}

	DEBUG("Enter the bytes to be transfered: ");
	fgets((char*)ptPck->tData.abWriteData, ptPck->tData.ulDataLength + 1, stdin);

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)ptPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_SendFactoryResetReq(APPLICATION_T* ptApp)
{
	APIOC_DCP_RESET_FACTORY_REQ_T tPck;
	TLR_UINT uiCnt = 0;

	char cBuf[128];

	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = PNIO_APCTL_CMD_DCP_RESET_FACTORY_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	DEBUG("\nThe MAC address of the IO-Device needs to be entered byte by byte in decimal format.\n");
	while (uiCnt < 6)
	{
		DEBUG("Enter byte number %u: ", uiCnt);
		fgets(cBuf, 128, stdin);
		tPck.tData.tMac[uiCnt] = atoi(cBuf);
		uiCnt++;
	}

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_ReadOutputData(APPLICATION_T* ptApp)
{
	char abData[32];
	TLR_UINT8 bData;
	TLR_UINT16 usData = 1000;
	TLR_RESULT tResult      = TLR_S_OK;

	while (usData > 255)
	{
		memset(abData, 0, 4);
		usData = 1000;
		DEBUG("Enter the byte (0...255) to be transfered and press Return: ");
		fgets(abData, 32, stdin);
		usData = atoi(abData);
	}
	bData = usData;
	DEBUG("writing 0x%02x to DPM\n", bData);

	tResult = xChannelIOWrite(ptApp->tCommon.hChannel, 0, 0, 1, &bData, 1);
	if (0 != tResult)
	{
		DEBUG("writing IO Data failed with 0x%08x\n", tResult);
	}
}

void App_GetHandles(APPLICATION_T* ptApp)
{
	RCX_PACKET_GET_SLAVE_HANDLE_REQ_T tPck;

	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = RCX_GET_SLAVE_HANDLE_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	tPck.tData.ulParam = RCX_LIST_CONF_SLAVES;


	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
	}
}

void App_GetIodInfo(APPLICATION_T* ptApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
	/* read MasterStatusBlock and printout configured IOD and active IOD */
	NETX_COMMON_STATUS_BLOCK tCommonStatus = {0};
	xChannelCommonStatusBlock(ptApp->tCommon.hChannel, CIFX_CMD_READ_DATA, 0, sizeof(tCommonStatus), &tCommonStatus);

	//DEBUG("configured IOD: %u\n", tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfConfigSlaves);
	if (ptApp->tCommon.ulConfigIodAmount != tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfConfigSlaves)
	{
		DEBUG("Configured IOD amount changed from %u to %u.\n", ptApp->tCommon.ulConfigIodAmount, tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfConfigSlaves);
		ptApp->tCommon.ulConfigIodAmount = tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfConfigSlaves;
	}

	//DEBUG("active IOD: %u\n", tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfActiveSlaves);
	if (ptApp->tCommon.ulActiveIodAmount != tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfActiveSlaves)
	{
			time_t rawtime;
			struct tm * timeinfo;
			time ( &rawtime );
			timeinfo = localtime ( &rawtime );

		DEBUG("%s", asctime(timeinfo));
		DEBUG("Active IOD amount changed from %u to %u.\n", ptApp->tCommon.ulActiveIodAmount, tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfActiveSlaves);
		ptApp->tCommon.ulActiveIodAmount = tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfActiveSlaves;
	}

	//DEBUG("faulty IOD: %u\n", tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfDiagSlaves);
	if (ptApp->tCommon.ulFaultyIodAmount != tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfDiagSlaves)
	{
		//DEBUG("Faulty IOD amount changed from %u to %u.\n", ptApp->tCommon.ulFaultyIodAmount, tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfDiagSlaves);
		ptApp->tCommon.ulFaultyIodAmount = tCommonStatus.uStackDepended.tMasterStatusBlock.ulNumOfDiagSlaves;
	}
}

void App_GetDeviceInfo(APPLICATION_T* ptApp, TLR_UINT32 ulHandle)
{
	RCX_PACKET_GET_SLAVE_CONN_INFO_REQ_T tPck;
	memset(&tPck, 0, sizeof(tPck));

	tPck.tHead.ulCmd  = RCX_GET_SLAVE_CONN_INFO_REQ;
	tPck.tHead.ulDest = 0x20;
	tPck.tHead.ulLen  = sizeof(tPck.tData);

	tPck.tData.ulHandle = ulHandle;

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tPck, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
	}
}



/*************************************************************************************************
 * @brief This method handles all packets which will be received by CoE application.
 * 
 * @param ptApp Pointer to application data.
 * @param ptPacket Pointer to CifX packet.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandlePacket(APPLICATION_T*  ptApp, CIFX_PACKET* ptPacket)
{
  TLR_PACKET_HEADER_T* ptPck = (TLR_PACKET_HEADER_T*)ptPacket;

  switch(ptPck->ulCmd)
  {
  case PNIO_APCTL_CMD_DCP_IDENT_ALL_CNF:
	  App_HandleIdentAllCnf(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_DCP_IDENT_IND:
	  App_HandleIdentEntryInd(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_DCP_IDENT_FINISHED_IND:
	  App_HandleIdentAllFinishInd(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_APPL_DIAG_DATA_IND:
	  App_HandleDiagnosisAlarmInd(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_APPL_ALARM_IND:
	  App_HandleAlarmInd(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_APPL_ALARM_ACK_CNF:
	  App_HandleAlarmAckCnf(ptApp, ptPacket);
	  break;

  case PNIO_APCTL_CMD_READ_CNF:
	  App_HandleReadCnf(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_READ_IMPL_CNF:
	  App_HandleReadImplCnf(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_WRITE_CNF:
	  App_HandleWriteCnf(ptApp, ptPacket);
	  break;

  case RCX_GET_SLAVE_HANDLE_CNF:
	  App_HandleGetSlaveHandleCnf(ptApp, ptPacket);
	  break;
  case RCX_GET_SLAVE_CONN_INFO_CNF:
	  App_HandleGetSlaveConnInfoCnf(ptApp, ptPacket);
	  break;
  case PNIO_APCTL_CMD_DCP_RESET_FACTORY_CNF:
	  App_HandleResetFactoryCnf(ptApp, ptPacket);
	  break;
    
  default:
      DEBUG("Unknown packet. ulCmd=0x%x\n", ptPacket->tHeader.ulCmd);
      break;
  }
}

/******************************************************************/
/*          functions handling packets send by firmware           */
/******************************************************************/

void App_HandleIdentAllCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_DCP_IDENT_ALL_CNF_T* ptPck = (APIOC_DCP_IDENT_ALL_CNF_T*)ptPacket;

	if (TLR_S_OK != ptPck->tHead.ulSta)
	{
		DEBUG("BusScan Confirmation: Error 0x%08x\n", ptPck->tHead.ulSta);
	}
}

void App_HandleIdentEntryInd(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_DCP_IDENT_ENTRY_IND_T* ptPck = (APIOC_DCP_IDENT_ENTRY_IND_T*)ptPacket;
	APIOC_DCP_IDENT_ENTRY_RSP_T* ptRsp = (APIOC_DCP_IDENT_ENTRY_RSP_T*)ptPacket;

	DEBUG("found IO-Device with MAC %02x:%02x:%02x:%02x:%02x:%02x, name \"%.*s\" and IP 0x%08x\n", 
		ptPck->tData.abMac[0], ptPck->tData.abMac[1], ptPck->tData.abMac[2], 
		ptPck->tData.abMac[3], ptPck->tData.abMac[4], ptPck->tData.abMac[5], 
		ptPck->tData.usLenName, ptPck->tData.abNameOfStation, ptPck->tData.ulIp);

	/* return the response */
	ptRsp->tHead.ulSta      = TLR_S_OK;
	ptRsp->tHead.ulCmd     |= 1;
	ptRsp->tHead.ulLen      = sizeof(ptRsp->tData);

	ptRsp->tData.ulReserved = 0;

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)ptRsp, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_HandleIdentAllFinishInd(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_DCP_IDENT_FINISHED_IND_T* ptPck = (APIOC_DCP_IDENT_FINISHED_IND_T*)ptPacket;
	APIOC_DCP_IDENT_FINISHED_RSP_T* ptRsp = (APIOC_DCP_IDENT_FINISHED_RSP_T*)ptPacket;

	if (TLR_S_OK != ptPck->tHead.ulSta)
	{
		DEBUG("BusScan Finished Indication: Error 0x%08x\n", ptPck->tHead.ulSta);
	}

	DEBUG("BusScan finished. Found %u IO-Devices.\n", ptPck->tData.ulDevCnt);
	
	/* return the response */
	ptRsp->tHead.ulSta      = TLR_S_OK;
	ptRsp->tHead.ulCmd     |= 1;
	ptRsp->tHead.ulLen      = 0;

	ptRsp->tData.ulReserved = 0;

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)ptRsp, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_HandleDiagnosisAlarmInd(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_DIAG_DATA_IND_T* ptPck = (APIOC_DIAG_DATA_IND_T*)ptPacket;
	APIOC_DIAG_DATA_RSP_T* ptRsp = (APIOC_DIAG_DATA_RSP_T*)ptPacket;

	TLR_UINT32 ulLen = ptPck->tHead.ulLen - sizeof(ptPck->tData) +1;
	DEBUG("Received diagnosis alarm from IOD %u: API %u, slot %u, subslot %u. USI 0x%08x and length %u.\n", ptPck->tData.ulHandle, ptPck->tData.ulApi, ptPck->tData.usSlot, ptPck->tData.usSubslot, ptPck->tData.usUserStructIdent, ulLen);
	
	/* return the response */
	ptRsp->tHead.ulSta      = TLR_S_OK;
	ptRsp->tHead.ulCmd     |= 1;
	ptRsp->tHead.ulLen      = 0;

	ptRsp->tData.ulHandle   = ptPck->tData.ulHandle;

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)ptRsp, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending packet failed.\n");
    }
}

void App_HandleAlarmInd(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_ALARM_IND_T* ptPck = (APIOC_ALARM_IND_T*)ptPacket;
	APIOC_ALARM_RSP_T* ptRsp = (APIOC_ALARM_RSP_T*)ptPacket;

	APIOC_ALARM_ACK_REQ_T tReq = {{0}};

	TLR_UINT32 ulLen = ptPck->tHead.ulLen - sizeof(ptPck->tData) +1;
	if (sizeof(ptPck->tData) > ptPck->tHead.ulLen)
	{
		/* some alarms do not have USI and data (e.g. Plug/Pull) */
		ulLen = 0;
		DEBUG("Received alarm (type %u) from IOD %u: API %u, slot %u, subslot %u.\n", ptPck->tData.usAlarmType, ptPck->tData.ulHandle, ptPck->tData.ulApi, ptPck->tData.usSlot, ptPck->tData.usSubslot);
	}
	else
	{
		DEBUG("Received alarm (type %u) from IOD %u: API %u, slot %u, subslot %u. USI 0x%08x and length %u.\n", ptPck->tData.usAlarmType, ptPck->tData.ulHandle, ptPck->tData.ulApi, ptPck->tData.usSlot, ptPck->tData.usSubslot, ptPck->tData.usUserStructIdent, ulLen);
	}
	
	/* return the response */
	ptRsp->tHead.ulSta      = TLR_S_OK;
	ptRsp->tHead.ulCmd     |= 1;
	ptRsp->tHead.ulLen      = 0;

	ptRsp->tData.ulHandle   = ptPck->tData.ulHandle;

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)ptRsp, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending response packet failed.\n");
    }

	/* generate the Alarm ACK */
	tReq.tHead.ulCmd  = PNIO_APCTL_CMD_APPL_ALARM_ACK_REQ;
	tReq.tHead.ulLen  = sizeof(tReq.tData);
	tReq.tHead.ulDest = 0x20;
	
	tReq.tData.ulHandle         = ptPck->tData.ulHandle;
	tReq.tData.usAlarmSpecifier = ptPck->tData.usAlarmSpecifier;
	tReq.tData.usReserved       = 0;

	if (CIFX_NO_ERROR != xChannelPutPacket(ptApp->tCommon.hChannel, (CIFX_PACKET *)&tReq, ptApp->tCommon.ulTimeout))
    {
      DEBUG("Sending request packet failed.\n");
    }
}

void App_HandleAlarmAckCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_ALARM_ACK_CNF_T* ptPck = (APIOC_ALARM_ACK_CNF_T*)ptPacket;

	if (TLR_S_OK != ptPck->tHead.ulSta)
	{
		DEBUG("AlarmAck Confirmation: Error 0x%08x\n", ptPck->tHead.ulSta);
	}
}

void App_HandleReadCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_READ_CNF_T* ptPck = (APIOC_READ_CNF_T*)ptPacket;

	if (TLR_S_OK != ptPck->tHead.ulSta || 0 != ptPck->tData.ulPnio)
	{
		DEBUG("Read Confirmation: rcX-Error 0x%08x, Profinet-Error 0x%08x\n", ptPck->tHead.ulSta, ptPck->tData.ulPnio);
	}
	else
	{
		DEBUG("Read on IOD %u, API %u, Slot %u, Subslot %u, Index %u returned %u byte:\n", ptPck->tData.ulHandle, ptPck->tData.ulApi, ptPck->tData.usSlot, ptPck->tData.usSubSlot, ptPck->tData.usIndex, ptPck->tData.ulDataLength);
		DEBUG("%s\n", ptPck->tData.abReadData);
	}
}

void App_HandleReadImplCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_READ_IMPL_CNF_T* ptPck = (APIOC_READ_IMPL_CNF_T*)ptPacket;

	if (TLR_S_OK != ptPck->tHead.ulSta || 0 != ptPck->tData.ulPnio)
	{
		DEBUG("ReadImpl Confirmation: rcX-Error 0x%08x, Profinet-Error 0x%08x\n", ptPck->tHead.ulSta, ptPck->tData.ulPnio);
	}
	else
	{
		DEBUG("ReadImpl API %u, Slot %u, Subslot %u, Index %u returned %u byte:\n", ptPck->tData.ulApi, ptPck->tData.usSlot, ptPck->tData.usSubSlot, ptPck->tData.usIndex, ptPck->tData.ulDataLength);
		DEBUG("%s\n", ptPck->tData.abReadData);
	}
}

void App_HandleWriteCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	APIOC_WRITE_CNF_T* ptPck = (APIOC_WRITE_CNF_T*)ptPacket;

	if (TLR_S_OK != ptPck->tHead.ulSta || 0 != ptPck->tData.ulPnio)
	{
		DEBUG("Write Confirmation: rcX-Error 0x%08x, Profinet-Error 0x%08x\n", ptPck->tHead.ulSta, ptPck->tData.ulPnio);
	}
	else
	{
		DEBUG("Write on IOD %u, API %u, Slot %u, Subslot %u, Index %u succeeded.\n", ptPck->tData.ulHandle, ptPck->tData.ulApi, ptPck->tData.usSlot, ptPck->tData.usSubSlot, ptPck->tData.usIndex);
	}
}

void App_HandleGetSlaveHandleCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	RCX_PACKET_GET_SLAVE_HANDLE_CNF_T* ptPck = (RCX_PACKET_GET_SLAVE_HANDLE_CNF_T*)ptPacket;

	/* calculate the amount of handles inside this packet */
	/* the first handle is ptPck->tData.aulHandle[0] */
	/* the second is (and so on) ptPck->tData.aulHandle[1] */
	TLR_UINT32 ulHandleCnt = ptPck->tHead.ulLen - sizeof(ptPck->tData);
	ulHandleCnt = ulHandleCnt / 4;
	ulHandleCnt++;
	TLR_UINT32 ulTmp;

	if (16 > ulHandleCnt)
	{
		for(ulTmp = 0; ulTmp < ulHandleCnt; ulTmp++)
		{
			/* get device infor for the specific handle */
			App_GetDeviceInfo(ptApp, ptPck->tData.aulHandle[ulTmp]);
		}
	}
	else
	{
		DEBUG("%u devices configured. This needs more precise handling as only 16 message can be put in the mailbox per time.\n", ulHandleCnt);
	}

}

void App_HandleGetSlaveConnInfoCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	RCX_PACKET_GET_SLAVE_CONN_INFO_CNF_T* ptPck = (RCX_PACKET_GET_SLAVE_CONN_INFO_CNF_T*)ptPacket;
	APIOC_INACTIVE_DEVICE_CONNECT_INFO_T* ptDataInactive;
	APIOC_ACTIVE_DEVICE_CONNECT_INFO_T*   ptDataActive;

	if (ptPck->tData.ulStructID == APIOC_INACTIVE_DEVICE_CONNECT_INFO_STRUCT_ID)
	{
		/* this means that the IO-Device is not in cyclic data exchange */
		/* only the configured parameters are returned by firmware */
		ptDataInactive = (APIOC_INACTIVE_DEVICE_CONNECT_INFO_T*)(&ptPck->tData +1);
		DEBUG("Handle %u (inactive): Name %s\n", ptPck->tData.ulHandle, ptDataInactive->abName);
	}
	else if (ptPck->tData.ulStructID == APIOC_ACTIVE_DEVICE_CONNECT_INFO_STRUCT_ID)
	{
		/* this means that the IO-Device is in cyclic data exchange */
		/* some parameters are "live" and were returned by the device */
		ptDataActive = (APIOC_ACTIVE_DEVICE_CONNECT_INFO_T*)(&ptPck->tData +1);
		DEBUG("Handle %u (active): Name %s\n", ptPck->tData.ulHandle, ptDataActive->abName);
	}
	else
	{
		DEBUG("unexpected StructId %u returned.\n", ptPck->tData.ulStructID);
	}
}

void App_HandleResetFactoryCnf(APPLICATION_T* ptApp, CIFX_PACKET* ptPacket)
{
	if (TLR_S_OK != ptPacket->tHeader.ulState)
	{
	  DEBUG("Factory Reset failed with 0x%08x \n", ptPacket->tHeader.ulState);
	}
	else
	{
	  DEBUG("Factory Reset succeeded.\n");
	}
}



/*************************************************************************************************
 * @brief This function creates all resources for the application. It contains the Profinet configuration.
 * It allocates memory for the application data and returns a pointer to it.
 * App_FreeResources() must be called in order to free resources again.
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_CreateResources(APPLICATION_T** pptApp)
{
  DEBUG("Allocating resources...\n");

  /* return value of function */
  TLR_RESULT tResult      = TLR_S_OK;
  /* helping variable for length calculation */
  TLR_UINT   uiLenMemory = 0;

  /* at the end of the function we will return the pointer */
  *pptApp = NULL;

  /* allocate memory for application data */
  APPLICATION_T* ptApp = (APPLICATION_T*)malloc(sizeof(APPLICATION_T));
  memset(ptApp, 0, sizeof(APPLICATION_T));
  
  /* allocate memory for read / write buffers */
  ptApp->tCommon.ulReadBufferSize  = 256 * sizeof(TLR_UINT8);
  ptApp->tCommon.ulWriteBufferSize = 256 * sizeof(TLR_UINT8);

  ptApp->tCommon.pabReadBuffer  =  (TLR_UINT8*) malloc(ptApp->tCommon.ulReadBufferSize);
  ptApp->tCommon.pabWriteBuffer =  (TLR_UINT8*) malloc(ptApp->tCommon.ulWriteBufferSize);

  /* initialize the read and write buffer with zero */
  memset(ptApp->tCommon.pabReadBuffer,  0, ptApp->tCommon.ulReadBufferSize);
  memset(ptApp->tCommon.pabWriteBuffer, 0, ptApp->tCommon.ulWriteBufferSize);

  /* return the pointer */
  *pptApp = ptApp;

  DEBUG("Successful.\n");
  return tResult;
}

/*************************************************************************************************
 * @brief This function frees all resources created by App_CreateResources(). 
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_FreeResources(APPLICATION_T* ptApp)
{
  DEBUG("Free resources...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* free buffer resources */
  free(ptApp->tCommon.pabReadBuffer);
  free(ptApp->tCommon.pabWriteBuffer);

  /* free application data container */
  free(ptApp);

  DEBUG("Successful.\n");
  return tResult;
}

/*************************************************************************************************
 * @brief This method handles the cyclic process data exchange.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_UINT8 g_bInputData = 0;
void App_HandleProcessData(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* cyclic io handing happens here */
  /* read RTD-Data */
  tResult = xChannelIORead(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulReadBufferSize, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulReadBufferSize);

  /* the action depends on the return value we have received */
  if ((CIFX_DEV_EXCHANGE_FAILED == tResult) || (CIFX_DEV_EXCHANGE_TIMEOUT == tResult))
  {
    DEBUG("Read failed with 0x%08x\n", tResult);
  }
  else if (CIFX_DEV_NO_COM_FLAG == tResult)
  {
    //DEBUG("Read failed. Device is not communicating\n");
  }
  else if (CIFX_NO_ERROR == tResult)
  {
	if (g_bInputData != ptApp->tCommon.pabReadBuffer[0])
	{
		DEBUG("Input data changed from 0x%02x to 0x%02x\n", g_bInputData, ptApp->tCommon.pabReadBuffer[0]);
	}
	g_bInputData = ptApp->tCommon.pabReadBuffer[0];

    //DEBUG("Read succeeded. Copy Inputs back to Outputs.\n");
    
#if 0
	/* we copy the memory we have read to the memory we want to send, */
    /* because we just want to mirror the data */
    memcpy (ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulWriteBufferSize);     

    // write IO-Data 
    tResult = xChannelIOWrite(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulWriteBufferSize, ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.ulWriteBufferSize);
#endif
  }
  else
  {
    /* received unexpected failure */
    DEBUG("Read failed unexpected with 0x%08x\n", tResult);
  }

}

/*************************************************************************************************
 * @brief This function initializes the application. 
 * Objects will be creates and services will be registered inside.
 * App_Finalize() must be called in order to achieve a friendly shutdown of application.
 * 
 * @param ptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Initialize(APPLICATION_T* ptApp)
{
  DEBUG("Initializing application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* register on EcatEsm task (indications for status) */
  tResult = App_SendRecvEmptyPkt(ptApp, RCX_REGISTER_APP_REQ);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }  
  
  return tResult;
}


/*************************************************************************************************
 * @brief This method finalizes the application. 
 * It returns handles, aso.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Finalize(APPLICATION_T* ptApp)
{
  DEBUG("Shutdown application...\n");

  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* at first we empty the queue, than we start with the shutdown sequence below */
  while(TLR_S_OK == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(tRecvPkt), &tRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    App_HandlePacket(ptApp, &tRecvPkt);
  }

  /* unregister on Profinet IO-Controller */
  App_SendRecvEmptyPkt(ptApp, RCX_UNREGISTER_APP_REQ);

  /* perform ChannelInit on Profinet IO-Controller to stop any communication and delete configuration */
  App_SendRecvEmptyPkt(ptApp, RCX_CHANNEL_INIT_REQ);

  return tResult;
}

/*************************************************************************************************
 * @brief This function prompts a debug message at application startup.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_PromptIntro(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  DEBUG("\n");
  DEBUG("**********************************************************\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   application example for Profinet IO-Controller       *\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Copyright (c) Hilscher GmbH. All Rights Reserved.    *\n");
  DEBUG("*                                                        *\n");
  DEBUG("**********************************************************\n");
  DEBUG("\n");

  return tResult;
}

